<?php
declare(strict_types=1);

use App\Core\ModuleInterface;
use App\Core\Router;
use App\Core\Database;
use App\Core\ModuleConfig;
use App\Core\Request;
use App\Core\Response;

/**
 * Пример модуля по новому протоколу.
 *
 * - module.php возвращает объект, реализующий ModuleInterface
 * - module.json содержит манифест (описание, опции, маршруты)
 */
class SampleHelloModule implements ModuleInterface
{
    public function getSlug(): string
    {
        return 'sample_hello';
    }

    public function register(Router $router): void
    {
        // Публичный маршрут: /hello-module
        $router->get('/hello-module', function (Request $request, Response $response): void {
            $text = (string) ModuleConfig::getBySlug('sample_hello', 'greeting_text', 'Привет из модуля SampleHello!');

            header('Content-Type: text/html; charset=utf-8');

            echo '<div class="container py-4">';
            echo '<h1>Модуль SampleHello</h1>';
            echo '<p>' . htmlspecialchars($text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</p>';
            echo '</div>';
        });

        // Админ-маршрут: /admin/hello-module
        $router->get('/admin/hello-module', function (Request $request, Response $response): void {
            $text = (string) ModuleConfig::getBySlug('sample_hello', 'greeting_text', 'Привет из модуля SampleHello!');

            header('Content-Type: text/html; charset=utf-8');

            echo '<div class="container py-4">';
            echo '<h1>Админка SampleHello</h1>';
            echo '<p>Текущий текст приветствия:</p>';
            echo '<pre>' . htmlspecialchars($text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</pre>';
            echo '<p>Изменить можно в &laquo;Настройках модуля&raquo; через админку.</p>';
            echo '</div>';
        });

        // Пример использования хука: логируем логины пользователей
        \App\Core\Hook::register('onUserLogin', function (array $ctx): void {
            $user = $ctx['user'] ?? null;
            if (is_array($user)) {
                $email = $user['email'] ?? '[unknown]';
                error_log('SampleHelloModule: пользователь залогинился: ' . $email);
            }
        });
    }

    public function install(Database $db): void
    {
        // Здесь можно выполнить дополнительные SQL-миграции, если нужно.
        // Пока оставляем пустым — достаточно install.sql или логики админки.
    }

    public function uninstall(Database $db): void
    {
        // Здесь можно удалить свои таблицы/данные, если нужно.
    }

    public function enable(Database $db): void
    {
        // Вызывается при включении модуля (можно, например, инициализировать опции).
    }

    public function disable(Database $db): void
    {
        // Вызывается при выключении модуля.
    }
}

return new SampleHelloModule();
